/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Id: TransportLayer.h 5815 2013-10-02 09:07:58Z MichaelT $
   Last Modification:
    $Author: MichaelT $
    $Date: 2013-10-02 11:07:58 +0200 (Mi, 02 Okt 2013) $
    $Revision: 5815 $
   
   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: yes (define _UNICODE)
     WinCE        : no
 
   Description:
     Declaration of the "Transport Layer" Class, which handles the hilscher transport
     on a single pyhsical interface


   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     2         03.03.09    MT       Structural change to allow placing physical drivers
                                    in a dll
     
     1         25.09.08    PL       initial version
 
**************************************************************************************/

/////////////////////////////////////////////////////////////////////////////
/// \file TransportLayer.h
/// Declaration of the "Transport Layer" Class, which handles the hilscher transport
/// on a single pyhsical interface
/////////////////////////////////////////////////////////////////////////////

#pragma once

class CPhysicalInterface;
class CTransportLayer;
class CEndpoint;

#include "DeviceHandler.h"
#include "APIInterface.h"
#include "Mmsystem.h"

#define   NX_DEFAULT_NAME                     "cifX"

#define   CHANNEL_TYPE_XSYSDEVICE             0
#define   CHANNEL_TYPE_CHANNEL                1

#define   TRANSPORT_TO_TRANSFER               1000      /* Transfer packet timeout            */
#define   TRANSPORT_TO_QUERY_SERVER           1000      /* Query server information timeout   */
#define   TRANSPORT_TO_ESTABLISH_CONNECTION   2000      /* Establish connection timeout       */
#define   TRANSPORT_THREAD_TIMEOUT            1000      /* Start transport thread timeout     */

// Map of supported data types, This is per Target
typedef std::deque<HIL_TRANSPORT_PACKET*>      HIL_PACKET_QUEUE;

#ifdef _DEBUG
  #define ABORT_RECEIVE_TIMEOUT 0xFFFFFFFF
#else
  #define ABORT_RECEIVE_TIMEOUT TRANSPORT_TO_TRANSFER
#endif

///////////////////////////////////////////////////////////////////////////////////////////
/// \class CTransportLayer
/// Class handling the Hilscher transport on a single physical interface
///////////////////////////////////////////////////////////////////////////////////////////
class CTransportLayer
{
  // default constructor
  CTransportLayer(void);
  
public:
  CTransportLayer(CPhysicalInterface* pcInterface, CDeviceHandler* pcDeviceHandler);
  virtual ~CTransportLayer(void);

  typedef struct TARGET_SERVER_INFO_Ttag
  {
    std::string   szServerName;
    uint32_t      ulParallelServices;
    uint32_t      ulBufferSize;
    uint32_t      ulFeatures;
    bool          fIgnoreSequenceNr;

    std::vector<uint16_t> cvSupportedDataTypes;

  } TARGET_SERVER_INFO_T;
  
  // Called by interface -> Init / Deinit transport layer
  int32_t                       Init  ( void);
  int32_t                       DeInit( void);
  
  // Recevice data from connector 
  void                          ReceiveData     ( uint8_t* pbData, uint32_t ulDataLen);
  
  // Helper function to generate a packet header
  void                          GeneratePacketHeader(PHIL_TRANSPORT_HEADER ptHeader);
  
  // Send a transport packet
  int32_t                       SendPacket      ( PHIL_TRANSPORT_PACKET       ptPacket, 
                                                  CDataLayer*                 pcLayer   = NULL,
                                                  uint32_t                    ulTimeout = 0,
                                                  bool                        fIgnoreActiveServiceCnt = false);
  
  int32_t                       TransferPacket  ( WORD                        usDataType,
                                                  uint8_t*                    pbSendData,
                                                  uint32_t                    ulSendDataLen,
                                                  uint8_t*&                   pbRecvData,
                                                  uint32_t&                   ulRecvDataLen,
                                                  CEndpoint*                  pcEndpoint,
                                                  uint8_t                     bChannel,
                                                  uint32_t                    ulTimeout);

  int32_t                       TransferPacket  ( PHIL_TRANSPORT_PACKET       ptSendPacket, 
                                                  PHIL_TRANSPORT_PACKET       ptRecvPacket,
                                                  uint32_t                    ulTimeout,
                                                  bool                        fIgnoreActiveServiceCnt = false);

  // Handle packet memory
  static HIL_TRANSPORT_PACKET*  AllocateTransportFrame(uint32_t ulDataSize);
  static void                   FreeTransportFrame(HIL_TRANSPORT_PACKET* pPacket);

  bool                          GetSendMbxState(uint32_t ulDevice, uint32_t& ulSendMBXState);
 
  int32_t                       QueryServerFeatures(uint32_t ulTimeout);  
  int32_t                       QueryDeviceInformation(  uint16_t             usDataType,
                                                         uint32_t             ulOptions, 
                                                         BYTE                 bDevice, 
                                                         BYTE                 bChannel,
                                                         uint32_t             ulBufferLen,
                                                         void*                pvBuffer,
                                                         uint32_t             ulTimeout);

  TARGET_SERVER_INFO_T*         GetServerInfo(void) { return &m_tServerInfo; }
  void                          GetInterfaceName(std::string& szName);
  CPhysicalInterface*           GetInterface(void)  { return  m_pcInterface; }
                                                                        


protected:
  
  /* Receive state machine data */
  typedef enum TRANSPORT_RX_STATE_Etag
  {
    eWAIT_FOR_COOKIE,
    eWAIT_FOR_HEADER,
    eWAIT_FOR_PACKETCOMPLETE,
    ePACKET_COMPLETE
  } TRANSPORT_RX_STATE_E;

  // Struct for the current receviced packet, used in recevice data function
  typedef struct CUR_RX_DATA_Ttag
  {
    TRANSPORT_RX_STATE_E  eRxState;
    HIL_TRANSPORT_PACKET  tPacket;
    uint32_t         ulHeaderOffset;    
    uint32_t         ulDataOffset;
  } CUR_RX_DATA_T;

  typedef enum
  {
    eKEEP_ALIVE_UNSUPPORTED,     /*!< No keep-alive support by device */
    eKEEP_ALIVE_INITIALIZATION,  /*!< Initialization phase            */
    eKEEP_ALIVE_ACTIVE,          /*!< Keep alive active               */
    eKEEP_ALIVE_TIMEOUT,         /*!< Keep alive times out            */

  } KEEPALIVE_STATE_E;

  CPhysicalInterface*                       m_pcInterface;                        /*!< Physical interface driver                                    */ 
  CDeviceHandler*                           m_pcDeviceHandler;                    /*!< Device handler object, to handle interface and device access */

  // Transfer syncronisation, by sequence and transaction number
  CRITICAL_SECTION                          m_tcsHeader;
  uint8_t                                   m_bSequenceNr;                        /*!< Number for assigning Request <--> Acknowledge                  */
  uint16_t                                  m_usTransactionNr;                    /*!< Number for assigning Request <--> Answer                       */
  HIL_TRANSPORT_HEADER                      m_tACKPacket;                         /*!< Packet for sending acknowledge                                 */

  HIL_TRANSPORT_PACKET                      m_tKeepAlivePacket;                   /*!< Packet for sending keep-alive requests                         */
  KEEPALIVE_STATE_E                         m_eKeepAliveState;
  uint32_t                                  m_ulKeepAliveIdent;
  uint32_t                                  m_ulLastKeepAlive;
  uint32_t                                  m_ulKeepAliveTimeout;
  uint32_t                                  m_ulSendTimeout;
  HANDLE                                    m_hKeepAliveThread;
  HANDLE                                    m_hKeepAliveEvent;
  HANDLE                                    m_hKeepAliveStop;

  static DWORD WINAPI KeepAliveThread(void* pvParam);

  TARGET_SERVER_INFO_T                      m_tServerInfo;

  /* Transport Thread */
  LONG                                      m_lRefCount;
  HANDLE                                    m_hTransportThread;
  HANDLE                                    m_hRxSemaphore;
  HANDLE                                    m_hTxSemaphore;
  HANDLE                                    m_hTransportThreadStop;

  typedef struct TRANSFER_INFO_Ttag
  {
    PHIL_TRANSPORT_PACKET ptSendPacket;
    HIL_TRANSPORT_PACKET  tRecvPacket;
    HANDLE                hWaitEvent;
    bool                  fAborted;

  } TRANSFER_INFO_T;

  // In service entry structure 
  typedef struct INSERV_ENTRY_Ttag
  {
    BYTE*                   pbAckState;
    HANDLE                  hWaitAckEvent;
    CDataLayer*             pcLayer;
    HIL_TRANSPORT_HEADER    tPktHeader;

  } INSERV_ENTRY_T;

  // Map of inservice packets <bSeq, INSERV_ENTRY_T>
  typedef std::map<BYTE, INSERV_ENTRY_T>                  HIL_INSERV_MAP;


  CRITICAL_SECTION                          m_tcsTransferLock;  /*!< Critical section to synchronize Timeout handling with data reception */
  std::vector<TRANSFER_INFO_T*>             m_cvPendingTransfers;

  bool                                      HandlePendingTransfers(PHIL_TRANSPORT_PACKET ptPacket);
  void                                      HandleKeepAlive(void);
  bool                                      AddParallelService(uint32_t ulTimeout, bool fIgnoreActiveServiceCnt = false );
  void                                      RemoveParallelService(void);
  void                                      HandleParallelServiceTimeout(void);

  CRITICAL_SECTION                          m_tcsRxLock;  /*!< Critical section to synchronize Timeout handling with data reception */
  CRITICAL_SECTION                          m_tcsRxQueue; /*!< Critical section to synchronize RX queue accesses */
  HIL_PACKET_QUEUE                          m_cRxQueue;
  CUR_RX_DATA_T                             m_tRxCurrent;                         /*!< Receive state machine                                          */
  uint32_t                                  m_ulLastRxTime;                       /*!< Last time data was received                                    */

  CRITICAL_SECTION                          m_tcsTxQueue; /*!< Critical section to synchronize RX queue accesses */
  std::deque<PHIL_TRANSPORT_PACKET>         m_cTxQueue;

  CRITICAL_SECTION                          m_tcsInService; /*!< Critical section to synchronize RX queue accesses */
  HIL_INSERV_MAP                            m_cmInService;

  CRITICAL_SECTION                          m_tcsActiveServices; /*!< Critical section to synchronize active service list accesses */
  std::list<uint32_t>                       m_cvActiveServices;

  static DWORD WINAPI   TransportThread(void* pvParam);
  void                  TransportThreadFunc(void);
  
  // Packet handling 
  uint16_t                            CalcCrc16(uint8_t* pbData, uint32_t ulDataLen);
  uint16_t                            CalcCrcNetIC(uint8_t* pbData, uint32_t ulDataLen);
  void                                SendAcknowledge(PHIL_TRANSPORT_HEADER ptHeader, uint8_t bState);
};
